<?php
declare(strict_types=1);

session_start();

/* ============ Helpers ============ */

function h(string $v): string {
  return htmlspecialchars($v, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function fmt_mxn($v): string {
  $n = (float)($v ?? 0);
  if (!is_finite($n)) $n = 0.0;
  if ($n < 0) $n = 0.0;
  $n = round($n, 2);
  return 'MXN ' . number_format($n, 2, ',', '.');
}

function base_url_dir(): string {
  $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || (($_SERVER['SERVER_PORT'] ?? '') === '443');
  $scheme = $https ? 'https' : 'http';
  $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
  $script = $_SERVER['SCRIPT_NAME'] ?? '';
  $dir = rtrim(str_replace('\\', '/', dirname($script)), '/');
  return $scheme . '://' . $host . ($dir ? $dir : '');
}

function file_admin(): string { return __DIR__ . '/admin.json'; }
function file_pagos(): string { return __DIR__ . '/config.json'; }
function file_qr(): string { return __DIR__ . '/qr.png'; }
function file_visits(): string { return __DIR__ . '/visits.jsonl'; }
function file_visits_salt(): string { return __DIR__ . '/visits_salt.txt'; }
function file_leads(): string { return __DIR__ . '/leads.json'; }

function strip_bom(string $s): string {
  // Quita UTF-8 BOM si existe
  if (substr($s, 0, 3) === "\xEF\xBB\xBF") return substr($s, 3);
  // Quita otros BOM raros si vinieran
  return ltrim($s, "\xEF\xBB\xBF\xFE\xFF\xFF\xFE");
}

function read_json(string $path, $fallback) {
  if (!is_file($path)) return $fallback;
  $raw = @file_get_contents($path);
  if (!is_string($raw) || $raw === '') return $fallback;

  $raw = strip_bom($raw);
  $data = json_decode($raw, true);

  if (json_last_error() !== JSON_ERROR_NONE) return $fallback;
  return $data;
}

function write_json_pretty(string $path, $data): bool {
  $json = json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
  if (!is_string($json)) return false;
  return @file_put_contents($path, $json . "\n", LOCK_EX) !== false;
}

function now_iso(): string { return gmdate('c'); }
function iso_to_ts(string $iso): int {
  $t = strtotime($iso);
  return $t === false ? 0 : $t;
}

function ensure_visits_salt(): string {
  $path = file_visits_salt();
  if (is_file($path)) {
    $s = trim((string)@file_get_contents($path));
    if ($s !== '') return $s;
  }
  $salt = bin2hex(random_bytes(16));
  @file_put_contents($path, $salt . "\n", LOCK_EX);
  return $salt;
}

function is_authed(): bool {
  return isset($_SESSION['tb_admin']) && $_SESSION['tb_admin'] === true;
}

function require_auth(): void {
  if (!is_authed()) {
    header('Location: ' . base_url_dir() . '/admin.php');
    exit;
  }
}

function ensure_csrf(): string {
  if (!isset($_SESSION['tb_csrf']) || !is_string($_SESSION['tb_csrf']) || $_SESSION['tb_csrf'] === '') {
    $_SESSION['tb_csrf'] = bin2hex(random_bytes(16));
  }
  return $_SESSION['tb_csrf'];
}

function check_csrf(): void {
  $token = (string)($_POST['csrf'] ?? '');
  $ok = isset($_SESSION['tb_csrf']) && is_string($_SESSION['tb_csrf']) && hash_equals($_SESSION['tb_csrf'], $token);
  if (!$ok) {
    http_response_code(400);
    echo "CSRF inválido.";
    exit;
  }
}

/* ============ leads (leads.json) ============ */

function read_leads(): array {
  $data = read_json(file_leads(), []);
  return is_array($data) ? $data : [];
}

function write_leads(array $leads): bool {
  return write_json_pretty(file_leads(), $leads);
}

function find_lead_index(array $leads, string $id): ?int {
  foreach ($leads as $i => $l) {
    if (!is_array($l)) continue;
    if ((string)($l['id'] ?? '') === $id) return $i;
  }
  return null;
}

/* ============ visitas (visits.jsonl) ============ */

function read_visits_events(int $limit = 5000): array {
  $path = file_visits();
  if (!is_file($path)) return [];

  $events = [];
  try {
    $f = new SplFileObject($path, 'r');
    $f->setFlags(SplFileObject::DROP_NEW_LINE);
    foreach ($f as $line) {
      if (!is_string($line)) continue;
      $line = trim($line);
      if ($line === '') continue;
      $row = json_decode($line, true);
      if (!is_array($row)) continue;
      $events[] = $row;
      if (count($events) > $limit) array_shift($events);
    }
  } catch (Throwable $e) {
    return [];
  }

  return $events;
}

function filter_events_since(array $events, int $sinceTs): array {
  if ($sinceTs <= 0) return $events;
  $out = [];
  foreach ($events as $e) {
    if (!is_array($e)) continue;
    $ts = iso_to_ts((string)($e['ts'] ?? ''));
    if ($ts >= $sinceTs) $out[] = $e;
  }
  return $out;
}

function count_unique_vid(array $events): int {
  $seen = [];
  foreach ($events as $e) {
    if (!is_array($e)) continue;
    $vid = (string)($e['vid'] ?? '');
    if ($vid === '') continue;
    $seen[$vid] = true;
  }
  return count($seen);
}

function group_counts_by_day(array $events): array {
  $days = [];
  foreach ($events as $e) {
    if (!is_array($e)) continue;
    $ts = (string)($e['ts'] ?? '');
    $day = $ts !== '' ? substr($ts, 0, 10) : '';
    if ($day === '' || strlen($day) !== 10) continue;
    $days[$day] = ($days[$day] ?? 0) + 1;
  }
  ksort($days);
  return $days;
}

function group_counts_by_path(array $events): array {
  $paths = [];
  foreach ($events as $e) {
    if (!is_array($e)) continue;
    $p = (string)($e['p'] ?? '');
    if ($p === '' || strlen($p) > 600) continue;
    $paths[$p] = ($paths[$p] ?? 0) + 1;
  }
  arsort($paths);
  return $paths;
}

/* ============ config.json model ============ */

function default_pagos(): array {
  return [
    "title" => "Elige tu método de pago",
    "subtitle" => "Selecciona cómo quieres pagar.",
    "support_whatsapp_url" => "",
    "methods" => [
      [
        "label" => "Nequi",
        "description" => "Pago con Nequi",
        "url" => "",
        "id" => "nequi",
        "type" => "link"
      ],
      [
        "label" => "Tarjeta",
        "description" => "Crédito / débito",
        "url" => "",
        "id" => "tarjeta",
        "type" => "link"
      ],
      [
        "id" => "qr",
        "label" => "Pagar con QR",
        "description" => "Escanear código",
        "type" => "qr",
        "image" => "qr.png",
        "url" => ""
      ],
      [
        "id" => "llave",
        "label" => "Llave Bre-B",
        "description" => "Tuboleta Pass",
        "type" => "text",
        "url" => "0091346036"
      ]
    ]
  ];
}

function ensure_methods_shape(array $p): array {
  $d = default_pagos();
  if (!isset($p['methods']) || !is_array($p['methods'])) $p['methods'] = $d['methods'];

  // Asegura que exista el método llave (Bre-B) aunque el config.json sea viejo.
  $hasLlave = false;
  foreach ($p['methods'] as $m) {
    if (is_array($m) && (string)($m['id'] ?? '') === 'llave') { $hasLlave = true; break; }
  }
  if (!$hasLlave) $p['methods'][] = $d['methods'][3];
  return $p;
}

/**
 * Busca método por:
 * 1) id exacto
 * 2) label contiene (Nequi/Tarjeta)
 * 3) type === qr
 */
function find_method_index(array $p, string $want): ?int {
  $methods = $p['methods'] ?? null;
  if (!is_array($methods)) return null;

  // 1) id exacto
  foreach ($methods as $i => $m) {
    if (!is_array($m)) continue;
    if ((string)($m['id'] ?? '') === $want) return $i;
  }

  // 2) por label
  $w = strtolower($want);
  if ($want === 'nequi' || $want === 'tarjeta') {
    foreach ($methods as $i => $m) {
      if (!is_array($m)) continue;
      $label = strtolower((string)($m['label'] ?? ''));
      if ($label !== '' && strpos($label, $w) !== false) return $i;
    }
  }

  // 3) por type qr
  if ($want === 'qr') {
    foreach ($methods as $i => $m) {
      if (!is_array($m)) continue;
      if ((string)($m['type'] ?? '') === 'qr') return $i;
    }
  }

  return null;
}

function get_method(array $p, string $want): array {
  $idx = find_method_index($p, $want);
  if ($idx === null) return [];
  return is_array($p['methods'][$idx] ?? null) ? $p['methods'][$idx] : [];
}

function set_method_url(array &$p, string $want, string $url): void {
  $idx = find_method_index($p, $want);
  if ($idx === null) return;
  if (!isset($p['methods'][$idx]) || !is_array($p['methods'][$idx])) return;
  $p['methods'][$idx]['url'] = $url;
}

function set_qr_image(array &$p, string $image): void {
  $idx = find_method_index($p, 'qr');
  if ($idx === null) return;
  if (!isset($p['methods'][$idx]) || !is_array($p['methods'][$idx])) return;
  $p['methods'][$idx]['image'] = $image;
}

/* ============ Admin auth ============ */

$admin = read_json(file_admin(), []);
$hasAdmin = is_array($admin) && !empty($admin['password_hash']) && is_string($admin['password_hash']);

$action = (string)($_GET['a'] ?? '');
if ($action === 'logout') {
  $_SESSION = [];
  session_destroy();
  header('Location: ' . base_url_dir() . '/admin.php');
  exit;
}

if ($action === 'download_leads') {
  require_auth();
  header('Content-Type: application/json; charset=utf-8');
  header('Content-Disposition: attachment; filename="leads.json"');
  header('Cache-Control: no-store');
  $p = file_leads();
  if (is_file($p)) {
    readfile($p);
  } else {
    echo "[]";
  }
  exit;
}

/* ============ POST ============ */

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $mode = (string)($_POST['mode'] ?? '');

  if ($mode === 'setup') {
    $pass1 = (string)($_POST['password'] ?? '');
    $pass2 = (string)($_POST['password2'] ?? '');
    if ($pass1 === '' || $pass1 !== $pass2 || strlen($pass1) < 6) {
      $err = "La contraseña debe tener mínimo 6 caracteres y coincidir.";
    } else {
      $admin = ['password_hash' => password_hash($pass1, PASSWORD_DEFAULT), 'created_at' => now_iso()];
      write_json_pretty(file_admin(), $admin);
      $_SESSION['tb_admin'] = true;
      header('Location: ' . base_url_dir() . '/admin.php');
      exit;
    }
  }

  if ($mode === 'login') {
    $pass = (string)($_POST['password'] ?? '');
    if (!$hasAdmin) {
      $err = "Primero configura el admin.";
    } elseif (!password_verify($pass, (string)($admin['password_hash'] ?? ''))) {
      $err = "Contraseña incorrecta.";
    } else {
      $_SESSION['tb_admin'] = true;
      header('Location: ' . base_url_dir() . '/admin.php');
      exit;
    }
  }

  if ($mode === 'save_fields') {
    require_auth();
    check_csrf();

    $title = trim((string)($_POST['title'] ?? ''));
    $subtitle = trim((string)($_POST['subtitle'] ?? ''));
    $supportWhatsapp = trim((string)($_POST['support_whatsapp_url'] ?? ''));
    // Normaliza wa.me (evita problemas por mayúsculas o falta de https).
    if ($supportWhatsapp !== '') {
      $supportWhatsapp = preg_replace('/\\s+/', '', $supportWhatsapp);
      if (preg_match('~^wa\\.me/~i', $supportWhatsapp)) $supportWhatsapp = 'https://' . $supportWhatsapp;
      if (preg_match('~^https?://wa\\.me/~i', $supportWhatsapp)) {
        $supportWhatsapp = preg_replace('~^https?://wa\\.me~i', 'https://wa.me', $supportWhatsapp);
      }
      if (preg_match('~^https?://(www\\.)?whatsapp\\.com/~i', $supportWhatsapp)) {
        // deja como está (solo fuerza https y lowercase host)
        $supportWhatsapp = preg_replace('~^https?://(www\\.)?whatsapp\\.com~i', 'https://whatsapp.com', $supportWhatsapp);
      }
    }
    $urlNequi = trim((string)($_POST['url_nequi'] ?? ''));
    $urlTarjeta = trim((string)($_POST['url_tarjeta'] ?? ''));
    $urlQr = trim((string)($_POST['url_qr'] ?? ''));
    $llaveBreb = trim((string)($_POST['llave_breb'] ?? ''));

    $p = read_json(file_pagos(), default_pagos());
    if (!is_array($p)) $p = default_pagos();
    $p = ensure_methods_shape($p);

    if ($title !== '') $p['title'] = $title;
    if ($subtitle !== '') $p['subtitle'] = $subtitle;
    $p['support_whatsapp_url'] = $supportWhatsapp;

    set_method_url($p, 'nequi', $urlNequi);
    set_method_url($p, 'tarjeta', $urlTarjeta);
    set_method_url($p, 'qr', $urlQr);
    set_method_url($p, 'llave', $llaveBreb);

    $p['_updated_at'] = now_iso();

    if (!write_json_pretty(file_pagos(), $p)) {
      $err = "No se pudo guardar config.json (permisos).";
    } else {
      $msg = "Cambios guardados en config.json";
    }
  }

  if ($mode === 'upload_qr') {
    require_auth();
    check_csrf();

    if (!isset($_FILES['qr']) || !is_array($_FILES['qr'])) {
      $err = "No llegó el archivo.";
    } else {
      $tmp = (string)($_FILES['qr']['tmp_name'] ?? '');
      $name = (string)($_FILES['qr']['name'] ?? '');
      $size = (int)($_FILES['qr']['size'] ?? 0);
      $errCode = (int)($_FILES['qr']['error'] ?? UPLOAD_ERR_OK);

      if ($errCode !== UPLOAD_ERR_OK || $tmp === '' || !is_uploaded_file($tmp)) {
        $err = "Error subiendo el QR.";
      } elseif ($size <= 0 || $size > 2_500_000) {
        $err = "El QR debe pesar máximo 2.5MB.";
      } else {
        $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
        if (!in_array($ext, ['png', 'jpg', 'jpeg', 'webp'], true)) {
          $err = "Formato inválido (usa png/jpg/webp).";
        } else {
          if (!@move_uploaded_file($tmp, file_qr())) {
            $err = "No se pudo guardar qr.png (permisos).";
          } else {
            // actualiza config.json (cache-bust)
            $p = read_json(file_pagos(), default_pagos());
            if (!is_array($p)) $p = default_pagos();
            $p = ensure_methods_shape($p);

            $img = 'qr.png?ts=' . time();
            set_qr_image($p, $img);
            $p['_updated_at'] = now_iso();
            write_json_pretty(file_pagos(), $p);

            $msg = "QR actualizado.";
          }
        }
      }
    }
  }

  if ($mode === 'lead_done') {
    require_auth();
    check_csrf();
    $id = trim((string)($_POST['id'] ?? ''));
    $leads = read_leads();
    $idx = $id !== '' ? find_lead_index($leads, $id) : null;
    if ($idx === null) {
      $err = "No se encontró el registro.";
    } else {
      $leads[$idx]['status'] = 'done';
      $leads[$idx]['done_at'] = now_iso();
      write_leads($leads);
      $msg = "Marcado como completado.";
    }
  }

  if ($mode === 'lead_delete') {
    require_auth();
    check_csrf();
    $id = trim((string)($_POST['id'] ?? ''));
    $leads = read_leads();
    $idx = $id !== '' ? find_lead_index($leads, $id) : null;
    if ($idx === null) {
      $err = "No se encontró el registro.";
    } else {
      array_splice($leads, $idx, 1);
      write_leads($leads);
      $msg = "Registro eliminado.";
    }
  }

  if ($mode === 'clear_leads') {
    require_auth();
    check_csrf();
    $ok = true;
    $path = file_leads();
    if (is_file($path)) $ok = @unlink($path);
    $msg = $ok ? 'Registros borrados.' : 'No se pudo borrar leads.json (permisos).';
  }

  if ($mode === 'clear_visits') {
    require_auth();
    check_csrf();
    $ok = true;
    $path = file_visits();
    if (is_file($path)) $ok = @unlink($path);
    $msg = $ok ? 'Estadísticas borradas.' : 'No se pudo borrar visits.jsonl (permisos).';
  }
}

/* ============ Load fields for UI ============ */

$csrf = ensure_csrf();

// Estadísticas (solo admin logueado)
$visitsAll = [];
$visits7 = [];
$visits30 = [];
$visitsByDay14 = [];
$visitsTopPages = [];
$visitsLast = [];
$visitsTotal = 0;
$visitsUnique30 = 0;
$visitsUnique7 = 0;
if (is_authed()) {
  ensure_visits_salt(); // crea visits_salt.txt para que track.php lo reutilice
  $visitsAll = read_visits_events(15000);
  $visitsTotal = count($visitsAll);

  $now = time();
  $visits7 = filter_events_since($visitsAll, $now - (7 * 24 * 60 * 60));
  $visits30 = filter_events_since($visitsAll, $now - (30 * 24 * 60 * 60));
  $visitsUnique7 = count_unique_vid($visits7);
  $visitsUnique30 = count_unique_vid($visits30);

  $byDay = group_counts_by_day($visits30);
  $last14 = [];
  for ($i = 13; $i >= 0; $i--) {
    $d = gmdate('Y-m-d', $now - ($i * 24 * 60 * 60));
    $last14[$d] = (int)($byDay[$d] ?? 0);
  }
  $visitsByDay14 = $last14;

  $visitsTopPages = array_slice(group_counts_by_path($visits30), 0, 10, true);
  $visitsLast = array_slice(array_reverse($visitsAll), 0, 30);
}

$leadsAll = [];
$leadsPending = 0;
$leadsDone = 0;
if (is_authed()) {
  $leadsAll = read_leads();
  foreach ($leadsAll as $l) {
    if (!is_array($l)) continue;
    $st = (string)($l['status'] ?? 'pending');
    if ($st === 'done') $leadsDone++;
    else $leadsPending++;
  }
}
$p = read_json(file_pagos(), default_pagos());
if (!is_array($p)) $p = default_pagos();
$p = ensure_methods_shape($p);

$mNequi = get_method($p, 'nequi');
$mTarjeta = get_method($p, 'tarjeta');
$mQr = get_method($p, 'qr');
$mLlave = get_method($p, 'llave');

$titleVal = (string)($p['title'] ?? '');
$subtitleVal = (string)($p['subtitle'] ?? '');
$supportWhatsappVal = (string)($p['support_whatsapp_url'] ?? '');
$urlNequiVal = (string)($mNequi['url'] ?? '');
$urlTarjetaVal = (string)($mTarjeta['url'] ?? '');
$urlQrVal = (string)($mQr['url'] ?? '');
$llaveBrebVal = (string)($mLlave['url'] ?? '0091346036');

$qrUrl = base_url_dir() . '/qr.png?ts=' . time();

?><!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Admin | Pagos</title>
  <style>
    :root{--bg:#f6f7fb;--card:#fff;--text:#0f172a;--muted:rgba(15,23,42,.65);--b:rgba(15,23,42,.10);--p:#2563eb;--shadow:0 18px 50px rgba(2,6,23,.08);}
    *{box-sizing:border-box}
    body{margin:0;font-family:ui-sans-serif,system-ui,-apple-system,Segoe UI,Roboto,Arial;background:linear-gradient(180deg,#fff,var(--bg));color:var(--text)}
    .wrap{max-width:980px;margin:0 auto;padding:18px 14px 30px}
    .top{display:flex;justify-content:space-between;align-items:center;gap:12px}
    .brand{display:flex;align-items:center;gap:10px}
    .logo{width:36px;height:36px;border-radius:12px;background:linear-gradient(135deg,#2563eb,#06b6d4);box-shadow:0 12px 30px rgba(37,99,235,.18)}
    h1{font-size:16px;margin:0;font-weight:950}
    .sub{margin:2px 0 0;font-size:12px;color:var(--muted)}
    .btn{border:1px solid var(--b);background:rgba(255,255,255,.75);border-radius:12px;padding:9px 12px;font-weight:900;cursor:pointer;text-decoration:none;color:inherit;display:inline-flex;align-items:center;gap:8px}
    .btn-primary{border-color:rgba(37,99,235,.35);background:rgba(37,99,235,.10)}
    .btn-danger{border-color:rgba(225,29,72,.30);background:rgba(225,29,72,.08)}
    .card{background:var(--card);border:1px solid var(--b);border-radius:18px;box-shadow:var(--shadow);overflow:hidden;margin-top:14px}
    .head{padding:14px 16px;border-bottom:1px solid var(--b)}
    .title{margin:0;font-size:13px;font-weight:950}
    .body{padding:14px 16px}
    .row{display:grid;gap:8px;margin-bottom:12px}
    label{font-size:12px;color:var(--muted);font-weight:900}
    input{border:1px solid var(--b);border-radius:12px;padding:10px 12px;font-weight:850;font-family:inherit}
    .msg{margin:12px 0 0;padding:10px 12px;border:1px solid rgba(22,163,74,.25);background:rgba(22,163,74,.06);border-radius:14px}
    .err{margin:12px 0 0;padding:10px 12px;border:1px solid rgba(225,29,72,.25);background:rgba(225,29,72,.06);border-radius:14px}
    .grid{display:grid;grid-template-columns:1.2fr .8fr;gap:14px}
    @media (max-width: 900px){.grid{grid-template-columns:1fr}}
    .preview{border:1px solid var(--b);border-radius:16px;padding:12px;background:rgba(15,23,42,.02)}
    .preview img{width:100%;max-width:240px;border-radius:14px;border:1px solid var(--b);background:#fff}
    code{font-family:ui-monospace,SFMono-Regular,Menlo,Consolas,monospace;font-size:12px}

    /* ====== Collapsibles (mejor visual) ====== */
    details{border:1px solid var(--b);border-radius:16px;background:rgba(15,23,42,.02);overflow:hidden}
    details + details{margin-top:10px}
    summary{list-style:none;cursor:pointer;padding:10px 12px;font-weight:950;display:flex;align-items:center;gap:10px}
    summary::-webkit-details-marker{display:none}
    summary:before{content:"▸";font-size:12px;opacity:.8}
    details[open] summary:before{content:"▾"}
    .details-body{padding:10px 12px;border-top:1px solid var(--b);background:#fff}
  </style>
</head>
<body>
  <div class="wrap">
    <div class="top">
      <div class="brand">
        <div class="logo" aria-hidden="true"></div>
        <div>
          <h1>Admin Pagos</h1>
          <div class="sub">Ahora sí muestra los valores actuales (BOM FIX)</div>
        </div>
      </div>
      <?php if (is_authed()): ?>
        <a class="btn btn-danger" href="<?=h(base_url_dir() . '/admin.php?a=logout')?>">Salir</a>
      <?php endif; ?>
    </div>

    <?php if (isset($msg)): ?><div class="msg"><?=h((string)$msg)?></div><?php endif; ?>
    <?php if (isset($err)): ?><div class="err"><?=h((string)$err)?></div><?php endif; ?>

    <?php if (!$hasAdmin): ?>
      <div class="card">
        <div class="head"><div class="title">Configurar admin</div></div>
        <div class="body">
          <form method="post">
            <input type="hidden" name="mode" value="setup" />
            <div class="row">
              <label>Contraseña nueva</label>
              <input type="password" name="password" required minlength="6" />
            </div>
            <div class="row">
              <label>Repetir contraseña</label>
              <input type="password" name="password2" required minlength="6" />
            </div>
            <button class="btn btn-primary" type="submit">Crear admin</button>
          </form>
        </div>
      </div>

    <?php elseif (!is_authed()): ?>
      <div class="card">
        <div class="head"><div class="title">Ingresar</div></div>
        <div class="body">
          <form method="post">
            <input type="hidden" name="mode" value="login" />
            <div class="row">
              <label>Contraseña</label>
              <input type="password" name="password" required />
            </div>
            <button class="btn btn-primary" type="submit">Entrar</button>
          </form>
        </div>
      </div>

    <?php else: ?>

      <div class="grid">
        <section class="card">
          <div class="head"><div class="title">Estadísticas de visitantes</div></div>
          <div class="body">
            <div class="sub">Cuenta entradas a las páginas HTML que llamen a <code>track.php</code>.</div>

            <div style="display:grid;grid-template-columns:repeat(3,minmax(0,1fr));gap:10px;margin-top:10px">
              <div class="preview">
                <div style="font-weight:950">Total (últimos 15k)</div>
                <div style="font-size:22px"><?=h((string)$visitsTotal)?></div>
              </div>
              <div class="preview">
                <div style="font-weight:950">Últimos 7 días</div>
                <div style="font-size:22px"><?=h((string)count($visits7))?></div>
                <div class="sub">Únicos: <?=h((string)$visitsUnique7)?></div>
              </div>
              <div class="preview">
                <div style="font-weight:950">Últimos 30 días</div>
                <div style="font-size:22px"><?=h((string)count($visits30))?></div>
                <div class="sub">Únicos: <?=h((string)$visitsUnique30)?></div>
              </div>
            </div>

            <hr style="border:0;border-top:1px solid var(--b);margin:14px 0" />

            <!-- Collapsibles para tablas pesadas -->
            <details open>
              <summary>Visitas por día (últimos 14)</summary>
              <div class="details-body">
                <table style="width:100%;border-collapse:collapse">
                  <thead>
                    <tr>
                      <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 4px">Día (UTC)</th>
                      <th style="text-align:right;border-bottom:1px solid var(--b);padding:6px 4px">Visitas</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php foreach ($visitsByDay14 as $d => $c): ?>
                      <tr>
                        <td style="padding:6px 4px;border-bottom:1px solid var(--b)"><?=h((string)$d)?></td>
                        <td style="padding:6px 4px;border-bottom:1px solid var(--b);text-align:right"><?=h((string)$c)?></td>
                      </tr>
                    <?php endforeach; ?>
                  </tbody>
                </table>
              </div>
            </details>

            <details>
              <summary>Top páginas (últimos 30)</summary>
              <div class="details-body">
                <table style="width:100%;border-collapse:collapse">
                  <thead>
                    <tr>
                      <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 4px">Página</th>
                      <th style="text-align:right;border-bottom:1px solid var(--b);padding:6px 4px">Visitas</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php if (empty($visitsTopPages)): ?>
                      <tr><td colspan="2" class="sub" style="padding:6px 4px">Aún no hay datos.</td></tr>
                    <?php else: ?>
                      <?php foreach ($visitsTopPages as $p2 => $c): ?>
                        <tr>
                          <td style="padding:6px 4px;border-bottom:1px solid var(--b)"><code><?=h((string)$p2)?></code></td>
                          <td style="padding:6px 4px;border-bottom:1px solid var(--b);text-align:right"><?=h((string)$c)?></td>
                        </tr>
                      <?php endforeach; ?>
                    <?php endif; ?>
                  </tbody>
                </table>
              </div>
            </details>

            <details>
              <summary>Últimas 30 entradas</summary>
              <div class="details-body">
                <table style="width:100%;border-collapse:collapse">
                  <thead>
                    <tr>
                      <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 4px">Fecha (UTC)</th>
                      <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 4px">Página</th>
                      <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 4px">Referer</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php if (empty($visitsLast)): ?>
                      <tr><td colspan="3" class="sub" style="padding:6px 4px">Aún no hay datos.</td></tr>
                    <?php else: ?>
                      <?php foreach ($visitsLast as $e): ?>
                        <tr>
                          <td style="padding:6px 4px;border-bottom:1px solid var(--b)"><?=h((string)($e['ts'] ?? ''))?></td>
                          <td style="padding:6px 4px;border-bottom:1px solid var(--b)"><code><?=h((string)($e['p'] ?? ''))?></code></td>
                          <td style="padding:6px 4px;border-bottom:1px solid var(--b)"><code><?=h((string)($e['r'] ?? ''))?></code></td>
                        </tr>
                      <?php endforeach; ?>
                    <?php endif; ?>
                  </tbody>
                </table>
              </div>
            </details>

            <hr style="border:0;border-top:1px solid var(--b);margin:14px 0" />

            <div style="display:flex;gap:10px;flex-wrap:wrap;align-items:center">
              <form method="post" onsubmit="return confirm('¿Borrar todas las estadísticas?');" style="margin:0">
                <input type="hidden" name="mode" value="clear_visits" />
                <input type="hidden" name="csrf" value="<?=h($csrf)?>" />
                <button class="btn btn-danger" type="submit">Borrar estadísticas</button>
              </form>
              <div class="sub">Archivo: <code>visits.jsonl</code> (se crea cuando alguien visita)</div>
            </div>
          </div>
        </section>

        <section class="card">
          <div class="head"><div class="title">Editar URLs (se guarda en config.json)</div></div>
          <div class="body">
            <form method="post">
              <input type="hidden" name="mode" value="save_fields" />
              <input type="hidden" name="csrf" value="<?=h($csrf)?>" />

              <div class="row">
                <label>Título (actual)</label>
                <input name="title" value="<?=h($titleVal)?>" />
              </div>

              <div class="row">
                <label>Subtítulo (actual)</label>
                <input name="subtitle" value="<?=h($subtitleVal)?>" />
              </div>

              <hr style="border:0;border-top:1px solid var(--b);margin:14px 0" />

              <div class="row">
                <label>URL Nequi (actual)</label>
                <input name="url_nequi" value="<?=h($urlNequiVal)?>" placeholder="https://checkout.bold.co/payment/..." />
              </div>

              <div class="row">
                <label>WhatsApp soporte (actual)</label>
                <input name="support_whatsapp_url" value="<?=h($supportWhatsappVal)?>" placeholder="https://wa.me/57..." />
              </div>

              <div class="row">
                <label>URL Tarjeta (actual)</label>
                <input name="url_tarjeta" value="<?=h($urlTarjetaVal)?>" placeholder="https://checkout.bold.co/payment/..." />
              </div>

              <div class="row">
                <label>URL QR (actual)</label>
                <input name="url_qr" value="<?=h($urlQrVal)?>" placeholder="https://checkout.bold.co/payment/..." />
              </div>

              <div class="row">
                <label>Llave Bre-B (Tuboleta Pass)</label>
                <input name="llave_breb" value="<?=h($llaveBrebVal)?>" placeholder="0091346036" />
              </div>

              <button class="btn btn-primary" type="submit">Guardar cambios</button>
            </form>
          </div>
        </section>

        <section class="card">
          <div class="head"><div class="title">Datos recibidos (Nombre/Email/WhatsApp)</div></div>
          <div class="body">
            <div style="display:flex;gap:10px;flex-wrap:wrap;align-items:center;margin-bottom:10px">
              <span class="btn" style="cursor:default">Pendientes: <strong><?=h((string)$leadsPending)?></strong></span>
              <span class="btn" style="cursor:default">Completados: <strong><?=h((string)$leadsDone)?></strong></span>
              <a class="btn" href="<?=h(base_url_dir() . '/admin.php?a=download_leads')?>">Descargar leads.json</a>
              <form method="post" onsubmit="return confirm('¿Borrar todos los registros?');" style="margin:0">
                <input type="hidden" name="mode" value="clear_leads" />
                <input type="hidden" name="csrf" value="<?=h($csrf)?>" />
                <button class="btn btn-danger" type="submit">Borrar registros</button>
              </form>
            </div>

            <details open>
              <summary>Ver tabla de registros</summary>
              <div class="details-body" style="padding:0">
                <div style="overflow:auto;max-height:520px;border-top:1px solid var(--b)">
                  <table style="width:100%;border-collapse:collapse">
                    <thead>
                      <tr>
                        <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 8px">Fecha (UTC)</th>
                        <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 8px">Cliente</th>
                        <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 8px">Boleta</th>
                        <th style="text-align:right;border-bottom:1px solid var(--b);padding:6px 8px">Total (MXN)</th>
                        <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 8px">Estado</th>
                        <th style="text-align:left;border-bottom:1px solid var(--b);padding:6px 8px">Acciones</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php if (empty($leadsAll)): ?>
                        <tr><td colspan="6" class="sub" style="padding:10px 8px">Aún no hay registros. Se crean cuando alguien envía sus datos desde <code>resumen.html</code> o <code>pasarela.html</code>.</td></tr>
                      <?php else: ?>
                        <?php foreach ($leadsAll as $l): if (!is_array($l)) continue; ?>
                          <?php
                            $st = (string)($l['status'] ?? 'pending');
                            $isDone = $st === 'done';
                            $client = trim((string)($l['name'] ?? ''));
                            $mail = trim((string)($l['email'] ?? ''));
                            $wa = trim((string)($l['whatsapp'] ?? ''));
                            $ticket2 = trim((string)($l['ticket'] ?? ''));
                            $evt2 = trim((string)($l['event'] ?? ''));
                            $date2 = trim((string)($l['date'] ?? ''));
                            $qty2 = (int)($l['qty'] ?? 1);
                            $addonLabel2 = trim((string)($l['addon_label'] ?? ''));
                            $addonQty2 = (int)($l['addon_qty'] ?? 0);
                            $addonTotal2 = (float)($l['addon_total'] ?? 0);
                            $total2 = (float)($l['total'] ?? 0);
                            $id2 = (string)($l['id'] ?? '');
                          ?>
                          <tr style="<?= $isDone ? 'opacity:.75' : '' ?>">
                            <td style="padding:8px;border-bottom:1px solid var(--b)"><?=h((string)($l['ts'] ?? ''))?></td>
                            <td style="padding:8px;border-bottom:1px solid var(--b)">
                              <div style="font-weight:950"><?=h($client)?></div>
                              <div class="sub"><?=h($mail)?></div>
                              <div class="sub"><?=h($wa)?></div>
                            </td>
                            <td style="padding:8px;border-bottom:1px solid var(--b)">
                              <div style="font-weight:950"><?=h($ticket2)?> <span class="sub">(x<?=h((string)$qty2)?>)</span></div>
                              <div class="sub"><?=h($evt2)?></div>
                              <div class="sub"><?=h($date2)?></div>
                              <?php if ($addonTotal2 > 0 && $addonLabel2 !== ''): ?>
                                <div class="sub">+ <?=h($addonLabel2)?> (x<?=h((string)$addonQty2)?>) — <?=h(fmt_mxn($addonTotal2))?></div>
                              <?php endif; ?>
                            </td>
                            <td style="padding:8px;border-bottom:1px solid var(--b);text-align:right;font-weight:950"><?=h(fmt_mxn($total2))?></td>
                            <td style="padding:8px;border-bottom:1px solid var(--b)">
                              <?= $isDone ? '<span class="btn" style="cursor:default">Completado</span>' : '<span class="btn btn-primary" style="cursor:default">Pendiente</span>' ?>
                            </td>
                            <td style="padding:8px;border-bottom:1px solid var(--b)">
                              <?php if (!$isDone): ?>
                                <form method="post" style="display:inline">
                                  <input type="hidden" name="mode" value="lead_done" />
                                  <input type="hidden" name="csrf" value="<?=h($csrf)?>" />
                                  <input type="hidden" name="id" value="<?=h($id2)?>" />
                                  <button class="btn btn-primary" type="submit">Marcar OK</button>
                                </form>
                              <?php endif; ?>
                              <form method="post" style="display:inline" onsubmit="return confirm('¿Eliminar este registro?');">
                                <input type="hidden" name="mode" value="lead_delete" />
                                <input type="hidden" name="csrf" value="<?=h($csrf)?>" />
                                <input type="hidden" name="id" value="<?=h($id2)?>" />
                                <button class="btn btn-danger" type="submit">Eliminar</button>
                              </form>
                            </td>
                          </tr>
                        <?php endforeach; ?>
                      <?php endif; ?>
                    </tbody>
                  </table>
                </div>
              </div>
            </details>
          </div>
        </section>

        <section class="card">
          <div class="head"><div class="title">QR (qr.png)</div></div>
          <div class="body">
            <div class="preview">
              <div style="font-weight:950;margin-bottom:8px">Vista previa</div>
              <img src="<?=h($qrUrl)?>" alt="QR" />
              <div class="sub" style="margin-top:8px">
                Archivo: <code>qr.png</code><br>
                URL: <code><?=h(base_url_dir() . '/qr.png')?></code>
              </div>
            </div>

            <form method="post" enctype="multipart/form-data" style="margin-top:12px">
              <input type="hidden" name="mode" value="upload_qr" />
              <input type="hidden" name="csrf" value="<?=h($csrf)?>" />
              <div class="row">
                <label>Subir QR (reemplaza <code>qr.png</code>)</label>
                <input type="file" name="qr" accept="image/png,image/jpeg,image/webp" required />
              </div>
              <button class="btn" type="submit">Subir QR</button>
            </form>

            <div class="sub" style="margin-top:10px">
              Al subir, se actualiza el método QR a <code>image: "qr.png?ts=..."</code> para evitar caché.
            </div>
          </div>
        </section>
      </div>

    <?php endif; ?>
  </div>
</body>
</html>
