<?php
declare(strict_types=1);

// Guarda datos del comprador (nombre/email/whatsapp) + resumen de la selección.
// Se consume desde resumen.html via fetch() antes de continuar al pago.

function now_iso(): string { return gmdate('c'); }

function no_cache(): void {
  header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
  header('Pragma: no-cache');
  header('Expires: 0');
}

function json_out(array $payload, int $code = 200): never {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  no_cache();
  echo json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
  exit;
}

function read_input(): array {
  $method = strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET'));
  if ($method !== 'POST') return [];

  $ct = strtolower((string)($_SERVER['CONTENT_TYPE'] ?? ''));
  if (str_contains($ct, 'application/json')) {
    $raw = file_get_contents('php://input');
    if (!is_string($raw) || trim($raw) === '') return [];
    $decoded = json_decode($raw, true);
    return is_array($decoded) ? $decoded : [];
  }

  return $_POST;
}

function clean_str($v, int $max = 200): string {
  $s = trim((string)$v);
  $s = str_replace(["\r", "\n", "\t"], ' ', $s);
  if (strlen($s) > $max) $s = substr($s, 0, $max);
  return $s;
}

function clean_phone(string $s): string {
  $s = trim($s);
  $s = preg_replace('/[^\d\+]/', '', $s);
  if (!is_string($s)) $s = '';
  if (strlen($s) > 20) $s = substr($s, 0, 20);
  return $s;
}

function clean_money($v): float {
  $n = (float)($v ?? 0);
  if (!is_finite($n)) $n = 0.0;
  if ($n < 0) $n = 0.0;
  // Evita valores absurdos.
  if ($n > 999999999) $n = 999999999.0;
  // 2 decimales (MXN).
  return round($n, 2);
}

function load_leads_locked($fh): array {
  rewind($fh);
  $raw = stream_get_contents($fh);
  if (!is_string($raw) || trim($raw) === '') return [];
  $data = json_decode($raw, true);
  return is_array($data) ? $data : [];
}

function save_leads_locked($fh, array $leads): bool {
  $json = json_encode($leads, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
  if (!is_string($json)) return false;
  rewind($fh);
  ftruncate($fh, 0);
  $ok = fwrite($fh, $json . "\n") !== false;
  fflush($fh);
  return $ok;
}

$in = read_input();
if (!$in) json_out(['ok' => false, 'error' => 'Método inválido'], 405);

$name = clean_str($in['name'] ?? '', 80);
$email = clean_str($in['email'] ?? '', 120);
$whatsapp = clean_phone(clean_str($in['whatsapp'] ?? '', 30));

$email = preg_replace('/[\\x00-\\x1F\\x7F\\x{200B}\\x{200C}\\x{200D}\\x{FEFF}]/u', '', $email);
$email = trim((string)$email);

if ($name === '' || strlen($name) < 2) json_out(['ok' => false, 'error' => 'Escribe tu nombre.'], 400);
if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) json_out(['ok' => false, 'error' => 'Correo inválido.'], 400);
if ($whatsapp === '' || strlen(preg_replace('/\D/', '', $whatsapp)) < 7) json_out(['ok' => false, 'error' => 'WhatsApp inválido.'], 400);

$entry = [
  'id' => bin2hex(random_bytes(8)),
  'ts' => now_iso(),
  'status' => 'pending',
  'name' => $name,
  'email' => $email,
  'whatsapp' => $whatsapp,
  'event' => clean_str($in['event'] ?? '', 160),
  'date' => clean_str($in['date'] ?? '', 160),
  'ticket' => clean_str($in['ticket'] ?? '', 200),
  'currency' => 'MXN',
  'unit_price' => clean_money($in['unit_price'] ?? 0),
  'qty' => (int)($in['qty'] ?? 1),
  'addon_label' => clean_str($in['addon_label'] ?? '', 120),
  'addon_unit_price' => clean_money($in['addon_unit_price'] ?? 0),
  'addon_qty' => (int)($in['addon_qty'] ?? 0),
  'addon_total' => clean_money($in['addon_total'] ?? 0),
  'total' => clean_money($in['total'] ?? 0),
  'page' => clean_str($in['page'] ?? ($_SERVER['HTTP_REFERER'] ?? ''), 400),
  'ref' => clean_str($in['ref'] ?? '', 400),
  'vid' => clean_str($_COOKIE['tb_vid'] ?? '', 40),
];

if ($entry['qty'] < 1) $entry['qty'] = 1;
if ($entry['qty'] > 10) $entry['qty'] = 10;
if ($entry['addon_qty'] < 0) $entry['addon_qty'] = 0;
if ($entry['addon_qty'] > 10) $entry['addon_qty'] = 10;
// money fields are already clamped via clean_money()

$path = __DIR__ . '/leads.json';
$fh = @fopen($path, 'c+');
if ($fh === false) json_out(['ok' => false, 'error' => 'No se pudo guardar (permisos).'], 500);

if (!flock($fh, LOCK_EX)) {
  fclose($fh);
  json_out(['ok' => false, 'error' => 'No se pudo guardar (lock).'], 500);
}

$leads = load_leads_locked($fh);
if (!is_array($leads)) $leads = [];

array_unshift($leads, $entry);
if (count($leads) > 2000) $leads = array_slice($leads, 0, 2000);

$ok = save_leads_locked($fh, $leads);
flock($fh, LOCK_UN);
fclose($fh);

if (!$ok) json_out(['ok' => false, 'error' => 'No se pudo guardar (write).'], 500);

json_out(['ok' => true, 'id' => $entry['id']]);
